﻿var dataTables = {
    itemDisplayMode: { selected: "selected", alwaysWithRows: "alwaysWithRows", always: "always" },
    filterDisplayMode: { buttons: "buttons", dropdown: "dropdown" }
};

(function ($) {
    var dataTableConstants = {
        maxFilterLength: 6,
        maxToolBarLength: 6,
        animationsDuration: 200,
        blockElementTimeout: 800,
        displayColumnsSelectorThreshold: 3,
        numericMinValue: 0,
        numericMaxValue: 2147483647
    }

    $.widget("a4.a4datatable", {
        options: {
            generateTableStructure: true,
            title: undefined,
            serverSide: true,
            ajaxAction: undefined,
            ajaxParams: {},
            data: [],
            columns: [],
            defaultSorting: [],
            deferRender: true,
            contextMenuItems: [],
            toolBarItems: [],
            quickFilter: undefined,
            localizableTexts: [],
            stateKey: undefined,
            allowRowSelection: undefined,
            rememberRowSelection: false,
            limitRowSelection: undefined,
            showSelectAll:true,
            selectedRows: false,
            displayColumnSelector: false,
            displaySearchBox: true,
            advancedFilter: false,
            displayEmptyTopContainer: false,
            style: "compact",
            displayLength: undefined,
            lengthMenu: [10, 20, 50, 100],
            tooltip: undefined,
            multiEdit: false,
            scrollY: '90vh',
            scrollCollapse: true,
            showPagination: true
            /*,autoBindColumns: false
            autoUpdate: undefined*/
        },
        _create: function () {
            var self = this;
            var opt = self.options;
            var loadStateCall = null;

            //Initializing container for methods that run before datatable is initialized
            this.container = this.element;

            if (opt.stateKey) {
                loadStateCall = this._loadState();
            }

            $.when(loadStateCall).then(function () {
                //Initializing parameters
                self.allowRowSelection = opt.allowRowSelection;

                if (self.allowRowSelection == undefined) {
                    self.allowRowSelection = _.some(opt.toolBarItems, function (t) {
                        return (!t.DisplayMode || t.DisplayMode == dataTables.itemDisplayMode.selected) && t.Visible != false;
                    });
                }

                self.updatedRows = [];

                //If datatable allows multi edit mode, it is disabled by default. Old grids only in edit mode, will work as expected
                if (opt.multiEdit) {
                    self.editing = false;
                }

                var lengthMenu = opt.lengthMenu;

                var pageLength = 20;

                //Custom page display Length
                if (opt.displayLength && _.isNumber(opt.displayLength)) {
                    lengthMenu.push(opt.displayLength);
                    lengthMenu.sort(function (a, b) { return a - b });
                    lengthMenu = _.uniq(lengthMenu, true);
                    pageLength = opt.displayLength;
                }

                if (self.state && self.state.PageLength) {
                    pageLength = self.state.PageLength;
                }

                self._initializeColumns();

                //Custom default sorting
                var defaultSorting = opt.defaultSorting;

                if (opt.defaultSorting && !_.isArray(opt.defaultSorting)) {
                    var defaultSortColumn = _.isObject(opt.defaultSorting) ? opt.defaultSorting.Column : opt.defaultSorting;

                    var defaultSortIndex = _.findIndex(self.columns, function (c) {
                        return c.name == defaultSortColumn;
                    });

                    var defaultSortDirecion = (opt.defaultSorting.Ascending === undefined || opt.defaultSorting.Ascending === true) ? "asc" : "desc";

                    if (defaultSortIndex > -1)
                        defaultSorting = [defaultSortIndex, defaultSortDirecion];
                }

                var table = self._renderTable();
                var tableDom = self._generateTableDom();

                var loadDataCall = null;

                //This object will be used in loadServerData. It will be merged with the filter params if a filter is applied.
                self.ajaxParams = self._getInitialAjaxParams();

                self.dataTable = table.DataTable({
                    scrollY: opt.scrollY,
                    scrollCollapse: opt.scrollCollapse,
                    paging:         true,
                    "pagingType": "full_numbers",
                    "searching": opt.displaySearchBox,
                    "processing": false,
                    "deferRender": opt.deferRender,
                    "serverSide": opt.serverSide,
                    "data": opt.data,
                    "ajax": function (data, callback, settings) {
                        if (opt.serverSide) {
                            if (opt.deferLoad) {
                                opt.deferLoad = false;
                            }
                            else {
                                loadDataCall = self._loadServerData(data, callback, settings);
                            }
                        }
                    },
                    "initComplete": function (settings, json) {
                    },
                    "drawCallback": function (settings) {
                        var api = this.api();
                        var pageInfo = api.page.info();
                        var container = api.table().container();

                        var selectAllCheckBox = $(".h-selectAll", container);

                        selectAllCheckBox.prop("checked", false);
                        selectAllCheckBox.toggle($(".selectRow", container).length > 0);

                        self._toggleToolbarItems(container);
                        self._checkSelectedRowsLimit();
                        self._setEditableItems();
                        self._toggleQuickFilter(container);

                        $(".dataTables_paginate", container).toggle(pageInfo.pages > 1);
                        $(".h-refreshButton").removeClass("fa-spin");

                        self._trigger("draw", null, { totalRecords: pageInfo.recordsTotal, displayedRecords: pageInfo.recordsDisplay });
                    },
                    "rowCallback": function (row, data) {
                        self._trigger("rowBeforeCreated", null, { row: row, data: data });
                    },
                    "createdRow": function (row, data, dataIndex) {
                        if (opt.rememberRowSelection && self.keyColumn && self.selectedRows && _.some(self.selectedRows, function (row) { return row[self.keyColumn] == data[self.keyColumn] })) {
                            $(row).addClass("selected");
                            $(row).find(".selectRow").attr("checked", "checked").prop("checked", true);
                        }
                        else if (!opt.serverSide) {
                            $(row).removeClass("selected");
                            $(row).find(".selectRow").removeClass("checked", "checked").prop("checked", false);
                        }

                        self._trigger("rowCreated", null, { index: dataIndex, row: row, data: data });
                    },
                    "footerCallback": function (tfoot, data, start, end, display) {
                        self._trigger("footerCreated", null, { footer: tfoot, data: data, start: start, end: end, display: display, additionalData: self.additionalData });
                    },
                    "columns": self.columns,
                    "order": defaultSorting,
                    "orderClasses": true,
                    "language": self._getDataTableLocalizableTexts(),
                    "dom": tableDom,
                    "autoWidth": false,
                    "pageLength": pageLength,
                    "lengthMenu": lengthMenu
                });

                //Updating container with the datatables container
                self.container = self.dataTable.table().container();

                self._renderAdvancedFilterToggleButton();
                self._renderMultiEditControls();
                self._renderSettingsMenu();
                self._renderToolBar();
                self._renderQuickFilter();
                self._bindEvents();
                self._initializeSelectedRows();

                $(".h-refreshButton", self.container).attr("title", self._getLocalizableText("Refresh"));

                if (opt.serverSide || !opt.data || opt.data.length == 0) {
                    $('.toolBar .button', self.container).hide();
                    $(".dataTables_paginate", self.container).hide();
                }

                if (opt.serverSide) {
                    //Hide table to prevent flickering
                    $(".dataTables-content", self.container).hide();
                    $(".dataTables-message", self.container).show();

                    //Waiting for the initialization calls to complete to display the table
                    $.when(loadDataCall).then(function () {
                        $(".dataTables-content", self.container).show("blind", "swing", dataTableConstants.animationsDuration);
                        $(".dataTables-message", self.container).hide();
                    }, function () {
                        $(".dataTables-message", self.container).addClass("error-message").html(self._getLocalizableText("UnexpectedError"));
                        self.hasError = true;
                    });
                }

                //Switching header row with advanced filter row
                if (opt.advancedFilter) {
                    $("thead tr.header-row", self.container).after($("thead tr.advanced-filter", self.container));
                }

                //Render title
                if (opt.title) {
                    $(".title", self.container).html(opt.title);
                }
            });
        },
        _loadServerData: function (data, callback, settings) {
            var self = this;
            var ajaxParams = { tableParams: this._getTableParams(data) };

            if (this.ajaxParams) {
                ajaxParams = _.extend(ajaxParams, this.ajaxParams);
            }

            // console.log("%c GetModulesList: folderId=" + ajaxParams.folderId + ", parentId=" + ajaxParams.parentId + ", " + "folderType=" + ajaxParams.folderType + ", " + "filterType=" + ajaxParams.filterType, 'background: #222; color: #bada55');

            return a4.callServerMethod(this.options.ajaxAction, ajaxParams,
            function (result) {
                if (result) {
                    if (result.d) {
                        result = _.isString(result.d) ? JSON.parse(result.d) : result.d;
                    }

                    self.additionalData = result.additionalData;
                    callback(result);
                }

                if (self.hasError) {
                    $(".dataTables-content", self.container).show();
                    $(".dataTables-message", self.container).hide();
                    self.hasError = false;
                }
            },
            function (error) {
                var message = "";

                try {
                    message = error.status == 404 ? error.statusText : eval("(" + error.responseText + ")").Message;
                }
                catch (ex) { }

                self.hasError = true;
                $(".dataTables-content", self.container).hide();
                $(".dataTables-message", self.container).addClass("error-message").html(message).show();

                self._trigger("error", null, { message: message });
            }, { blockElement: $(settings.nTable), blockElementTimeout: dataTableConstants.blockElementTimeout });
        },
        _setOption: function (key, value) {
            var options = this.options;
            switch (key) {
                case "ajaxParams":
                    if (!options.ajaxParams) {
                        options.ajaxParams = {};
                    }
                    //Merge with the user specified params
                    options.ajaxParams = _.extend(options.ajaxParams, value);

                    if (!this.ajaxParams) {
                        this.ajaxParams = {};
                    }
                    //Merge with the current grid params (user specified + filter params)
                    this.ajaxParams = _.extend(this.ajaxParams, value);
                    break;
                case "selectedRows":
                    options.selectedRows = value;
                    this._initializeSelectedRows();
                    this.refresh();
                    break;
                case "data":
                    options.data = value;
                    this.dataTable.clear();
                    this.dataTable.rows.add(options.data).draw();
                    this.updatedRows = [];
                    break;
                case "defaultSorting":
                    options.defaultSorting = value;
                    this.dataTable.order(options.defaultSorting);
                    break;
            }
        },
        destroy: function () {
            this.dataTable.destroy(true);
            $.Widget.prototype.destroy.call(this);
        },
        _loadState: function () {
            var self = this;
            return a4.callServerMethod(a4.getUrl("Shared/GetDataTableConfiguration"), { type: this.options.stateKey, itemId: 0 }, function (result) {
                if (result) {
                    self.state = result;
                }
            });
        },
        _saveState: function () {
            var settings = this.dataTable.settings();

            var configuration = {
                Columns: this.getSelectedColumns(),
                PageLength: settings.page.len(),
                Style: $(".h-densitySelector .button.selected", this.container).attr("data-display-density")
            };

            this.state = configuration;

            if (this.options.stateKey) {
                a4.callServerMethod(a4.getUrl("Shared/UpdateDataTableConfiguration"), { type: this.options.stateKey, itemId: 0, config: configuration }, function (result) { });
            }

            this._trigger("stateChange", null, configuration);
        },
        _toggleToolbarItems: function (container) {
            if (this.options.toolBarItems && this.options.toolBarItems.length > 0) {
                var toolBar = $(".toolBar", container);
                var filter = $(".filter-bar", container);
                var self = this;

                var selectedItems = this.getSelectedItems();

                if (this.editing) {
                    toolBar.hide();
                    filter.show();
                }
                else if (selectedItems && selectedItems.length > 0) {
                    if (this.options.quickFilter && !this.options.quickFilter.AlwaysVisible) {
                        filter.hide();
                    }

                    toolBar.show();

                    _.each(this.options.toolBarItems, function (item) {
                        var toolBarItem = $("[data-position=" + item.Position + "]", toolBar);
                        var visible = true;

                        if ((item.DisplayMode == undefined || item.DisplayMode == dataTables.itemDisplayMode.selected) && !item.Disabled) {
                            if (item.DisplayCondition) {
                                visible = _.every(selectedItems, function (row) {
                                    return a4.evaluateCondition(item.DisplayCondition, row);
                                });
                            }
                        }
                        else {
                            visible = false;
                        }

                        toolBarItem.toggle(visible);
                    });
                }
                else {
                    toolBar.show();
                    filter.show();

                    if ($(".dataTables_empty", container).length > 0) {
                        _.each(this.options.toolBarItems, function (item) {
                            $("[data-position=" + item.Position + "]", toolBar).toggle(item.DisplayMode == dataTables.itemDisplayMode.always);
                        });
                    }
                    else {
                        _.each(this.options.toolBarItems, function (item) {
                            $("[data-position=" + item.Position + "]", toolBar).toggle(item.DisplayMode != undefined && item.DisplayMode != dataTables.itemDisplayMode.selected);
                        });
                    }
                }
            }
        },
        _getDataTableLocalizableTexts: function () {
            return {
                lengthMenu: this._getLocalizableText("LengthMenu"),
                emptyTable: this._getLocalizableText("EmptyTable"),
                info: this._getLocalizableText("Info"),
                infoEmpty: this._getLocalizableText("InfoEmpty"),
                infoFiltered: this._getLocalizableText("InfoFiltered"),
                zeroRecords: this._getLocalizableText("ZeroRecords"),
                paginate: {
                    "first": "<span class='fa fa-lg fa-angle-double-left'></span>",
                    "previous": "<span class='fa fa-lg fa-angle-left'></span>",
                    "next": "<span class='fa fa-lg fa-angle-right'></span>",
                    "last": "<span class='fa fa-lg fa-angle-double-right'></span>"
                },
                search: "",
                loadingRecords: "",
                processing: ""
            };
        },
        _getLocalizableText: function (key) {
            var text = undefined;

            try {
                if (this.options.localizableTexts[key]) {
                    text = this.options.localizableTexts[key];
                }
                else if (dataTableResources && dataTableResources[key]) {
                    text = dataTableResources[key];
                }
                else if (sharedResources && sharedResources[key]) {
                    text = sharedResources[key];
                }
                else if (resources && resources[key]) {
                    text = resources[key];
                }
            }
            catch (err) { }

            return text;
        },
        _initializeColumns: function () {
            var columns = [];
            var self = this;

            //Select row column
            if (this.allowRowSelection) {
                columns.push({
                    name: "SelectRow",
                    data: function (source, type, val) {
                        var data = "";

                        if (type == "display" && a4.evaluateCondition(self.allowRowSelection, source)) {
                            data = "<input type='checkbox' class='selectRow' title='" + self._getLocalizableText("SelectRow") + "'>"
                        }

                        return data;
                    },
                    width: "25px",
                    orderable: false,
                    searchable: false,
                    IsDataColumn: false
                });
            }

            if (this.options.autoBindColumns) {
                self._autoBindColumns();
            }

            _.each(this.options.columns, function (col) {
                columns.push(self._createColumn(col));
            });

            //Context menu column
            if (this.options.contextMenuItems.length > 0) {
                columns.push({
                    name: "ActionsMenu",
                    data: null,
                    render: function (data, type, row, meta) {
                        var data = "";
                        if (type == "display") {
                            data = self._renderContextMenu(row);
                        }
                        return data;
                    },
                    width: "40px",
                    className: "action-menu-column",
                    orderable: false,
                    searchable: false,
                    IsDataColumn: false
                });
            }
            else if (this.options.advancedFilter) {
                columns.push({
                    name: "AdvancedFilterMenu",
                    data: null,
                    defaultContent: "",
                    width: "42px",
                    orderable: false,
                    searchable: false,
                    IsDataColumn: false
                });
            }

            if (this.options.tooltip) {
                var tooltipColumn = {};
                var tooltipData = this.options.tooltip.Data;
                var tooltipCondition = this.options.tooltip.DisplayCondition;
                var columnPosition = this.options.tooltip.Position || (this.allowRowSelection != false ? 1 : 0);

                tooltipColumn.className = "tooltip-column";
                tooltipColumn.name = "Tooltip";
                tooltipColumn.width = "10px";
                tooltipColumn.orderable = false;

                if (_.isFunction(tooltipData)) {
                    tooltipColumn.data = function (source, type, val) {
                        var visible = false;

                        if (type == "display") {
                            visible = tooltipCondition ? a4.evaluateCondition(tooltipCondition, source) : true;
                        }

                        return visible ? "<div class='clearfix'><span class='fa fa-warning'></span><div class='tooltip-content'></div></div>" : "";
                    };
                }
                else {
                    tooltipColumn.data = function (source, type, val) {
                        var visible = false;

                        if (type == "display" && source[tooltipData]) {
                            visible = tooltipCondition ? a4.evaluateCondition(tooltipCondition, source) : true;
                        }

                        return visible ? "<div class='clearfix'><span class='fa fa-warning'></span><div class='tooltip-content'>" + source[tooltipData] + "</div></div>" : "";
                    };
                }

                columns.splice(columnPosition, 0, tooltipColumn);
            }

            this.columns = columns;

            return columns;
        },
        _getInitialAjaxParams: function () {
            //A clone is needed to not mess with the parameters specified by the user.
            var ajaxParams = _.clone(this.options.ajaxParams);

            //Use the first filter params if there is a quick filter without the option All
            if (this.options.quickFilter && this.options.quickFilter.IncludeAll === false && this.options.quickFilter.Items && this.options.quickFilter.Items[0]) {
                ajaxParams = _.extend(ajaxParams, this.options.quickFilter.Items[0].AjaxParams);
            }

            return ajaxParams;
        },
        _autoBindColumns: function () {
            var tableParams = {
                Echo: 999,
                PageStart: 0,
                PageLength: 1,
                IsSearch: false,
                IsSort: false
            };

            ajaxParams = _.extend({ tableParams: tableParams }, this.options.ajaxParams);

            return a4.callServerMethod(this.options.ajaxAction, ajaxParams,
            function (result) {
                console.log(result);
            });
        },
        _createColumn: function (column) {
            var dataTableColumn = {};
            var sortable = (column.Sortable === undefined || column.Sortable) && !column.Index;
            var cssClass;
            var options = this.options;

            cssClass = "value-column";

            if (column.Class) {
                cssClass += " " + column.Class;
            }

            if (column.Name === undefined && column.Data !== undefined) {
                column.Name = column.Data;
            }
            else if (column.Data === undefined && column.Name !== undefined) {
                column.Data = column.Name;
            }

            var title = column.Title;

            if (!title) {
                title = this._getLocalizableText(column.Name);
            }

            dataTableColumn.title = "<span class='column-title'>" + title + "</span>" + (sortable ? "<span class='fa fa-caret-up'></span><span class='fa fa-caret-down'></span>" : "");
            dataTableColumn.width = column.Width;
            dataTableColumn.type = column.Type;
            dataTableColumn.orderable = sortable;
            dataTableColumn.searchable = column.Searchable !== false && this._isVisibleColumn(column);
            dataTableColumn.orderData = column.DataSort;
            dataTableColumn.visible = this._isVisibleColumn(column);
            dataTableColumn.data = column.Data;
            dataTableColumn.name = column.Name;
            dataTableColumn.defaultContent = column.DefaultValue;

            if (column.Editable) {
                var self = this;
                dataTableColumn.render = function (data, type, row, meta) {
                    var value = "";

                    if (a4.evaluateCondition(column.Editable, row)) {
                        var dataProperty = column.Data;

                        if (type == "display" && (self.editing == undefined || self.editing)) {
                            if (column.Type == "enum") {
                                input = self._generateEnumControl(column.EnumType, data, false);
                                dataProperty = column.Name;
                            }
                            else if (column.Type == "boolean") {
                                input = $("<input />", { "type": "checkbox" });
                                if (data) {
                                    input.attr("checked", "checked");
                                }
                            }
                            else if (column.Type == "richtext") {
                                input = $("<div />", { "class": "v-qeRichEditText", "html": data });
                            }
                            else {
                                input = $("<input />", { "type": "text", "value": data });
                            }

                            input.addClass("editable");
                            input.attr("data-dataproperty", dataProperty);

                            if (column.Type) {
                                input.attr("data-type", column.Type);
                            }

                            value = input[0].outerHTML;
                        }
                        else if (data != undefined) {
                            value = (column.Prefix || "") + data + (column.Suffix || "");
                        }
                    }
                    else {
                        value = data;
                    }

                    return value;
                }
            }
            else if (column.Label) { //Check need for this
                dataTableColumn.render = function (data, type, row, meta) {
                    return row[column.Label];
                }
            }
            else if (column.Prefix || column.Suffix) {
                dataTableColumn.render = function (data, type, row, meta) {
                    if (data) {
                        return (column.Prefix || "") + data + (column.Suffix || "");
                    }

                    return "";
                }
            }
            else if (column.Type == "boolean" && column.ShowIcon) {
                cssClass += " icon-column";

                dataTableColumn.render = function (data, type, row, meta) {
                    return data ? "<span class=\"fa fa-check\"></span>" : "";
                }
            }

            if (column.Index && !dataTableColumn.data) {
                dataTableColumn.data = function (row, type, set, meta) {
                    var index = meta.row + 1;

                    if (options.serverSide) {
                        index += meta.settings._iDisplayStart;
                    }

                    row.Index = index;

                    return index;
                };
            }

            if (column.Tooltip) {
                dataTableColumn.createdCell = function (cell, cellData, rowData, rowIndex, colIndex) {
                    if (rowData[column.Tooltip]) {
                        $(cell).attr("title", rowData[column.Tooltip].replace(/<(?:.|\n)*?>/gm, ''));
                    }
                };
            }

            if (column.Key) {
                this.keyColumn = column.Name;
            }

            dataTableColumn.className = cssClass;

            //Aditional properties            
            dataTableColumn.IsDataColumn = true;
            dataTableColumn.HideInSelector = column.HideInSelector;
            dataTableColumn.AdvancedFilter = column.AdvancedFilter;
            dataTableColumn.AdvancedFilterLabels = column.AdvancedFilterLabels;
            dataTableColumn.EnumType = column.EnumType;
            dataTableColumn.Editable = column.Editable;

            return dataTableColumn;
        },

        _getColumnByName: function (name) {
            return _.find(this.columns, function (column) { return column.name == name });
        },
        _isDataColumn: function (columnName) {
            var column = this._getColumnByName(columnName);

            if (column) {
                return column.IsDataColumn;
            }

            return false;
        },
        _isVisibleColumn: function (column) {
            var isVisible = column.Visible !== false;

            if (!column.HideInSelector && this.state && this.state.Columns && this.state.Columns.length > 0) {
                isVisible = _.some(this.state.Columns, function (c) { return c.toLowerCase() == column.Name.toLowerCase(); });
            }

            return isVisible;
        },
        _renderTable: function () {
            var table = this.container;
            var columns = this.columns;
            var self = this;

            var style = this.options.style || "";

            if (this.state && this.state.Style) {
                style = this.state.Style;
            }

            if (this.options.generateTableStructure) {
                table = $("<table />", { "class": style });

                var tableHead = $("<thead />");

                self._renderAdvancedFilterRow(tableHead);

                //Render header row
                var headerRow = $("<tr />", { "class": "header-row" });

                _.each(columns, function (column, index) {
                    var cell = $("<th />", { "data-column-name": column.Name });

                    if (index == 0 && self.allowRowSelection != false && !self.options.limitRowSelection && self.options.showSelectAll) {
                        cell.append($("<input />", { "type": "checkbox", "class": "h-selectAll", "title": self._getLocalizableText("SelectAllRows") }));
                    }

                    headerRow.append(cell);
                });

                //Render content row
                tableHead.append(headerRow);

                var tableBody = $("<tbody />");
                var contentRow = $("<tr />");

                _.each(columns, function (column) {
                    contentRow.append($("<td />"));
                });

                tableBody.append(contentRow);

                table.append(tableHead);
                table.append(tableBody);

                this.container.append(table);
            }
            else {
                if (this.options.advancedFilter) {
                    this._renderAdvancedFilterRow($("thead", this.container));
                }
                else {
                    $("th.advanced-filter, td.advanced-filter", this.container).detach();
                }
            }

            return table;
        },
        _generateTableDom: function () {
            var filterSection = "";
            var columnSelector = "";
            var advancedFilterToggleButton = "";
            var multiEditToggleButton = "";
            var multiEditBar = "";
            var toolBar = "";
            var filter = "";
            var top = "";
            var hasTop = this.options.displayEmptyTopContainer;
            var title = "";
            var tableContent = "";
            var opt = this.options;

            if (opt.displaySearchBox) {
                filterSection = 'f';
                hasTop = true;
            }

            if (opt.multiEdit) {
                multiEditToggleButton = '<"multi-edit-toggle">';
                multiEditBar = '<"multi-edit-bar">';
                hasTop = true;
            }

            if (opt.displayColumnSelector == true || (opt.displayColumnSelector == undefined && _.filter(opt.columns, function (c) { return c.Visible || !c.HideInSelector; }).length > dataTableConstants.displayColumnsSelectorThreshold)) {
                columnSelector = '<"settings-menu">';
                hasTop = true;
            }

            if (opt.advancedFilter) {
                advancedFilterToggleButton = '<"advanced-filter-toggle">';
                hasTop = true;
            }

            if (opt.toolBarItems.length > 0) {
                toolBar = '<"toolBar clearfix">';
                hasTop = true;
            }

            if (opt.quickFilter) {
                filter = '<"filter-bar">';
                hasTop = true;
            }

            if (opt.title) {
                title = '<"title">'
            }

            if (hasTop) {
                top = '<"top clearfix"' + filter + toolBar + columnSelector + multiEditToggleButton + advancedFilterToggleButton + filterSection + '>'
            }

            tableContent = '<"dataTables-content" ' + title + top + 't<"bottom clearfix no-print"i<"v-selectedRows selected-rows"><"h-refreshButton refresh-button fa fa-lg fa-refresh">' + (opt.showPagination ? 'pl' : '') + multiEditBar + '>>';

            if (opt.serverSide) {
                tableContent += '<"dataTables-message"<"loading-icon">>';
            }

            return tableContent;
        },
        _evaluateContextMenuDisplayCondition: function (menuItem, rowData) {
            var displayItem = false;

            if (rowData && menuItem && menuItem.DisplayCondition) {
                displayItem = a4.evaluateCondition(menuItem.DisplayCondition, rowData);
            }
            else {
                displayItem = (menuItem.Visible == undefined || menuItem.Visible);
            }

            return displayItem;
        },
        _renderContextMenu: function (row) {
            var contextMenu = $("<div />", { "class": "displayOnHover" });
            var self = this;

            var itemsToDisplay = _.filter(this.options.contextMenuItems, function (i) {
                var displayItem = self._evaluateContextMenuDisplayCondition(i, row);

                if (i.SubMenuItems) {
                    i.SubMenuItemsToDisplay = _.filter(i.SubMenuItems, function (j) { return self._evaluateContextMenuDisplayCondition(j, row); });

                    displayItem = displayItem && i.SubMenuItemsToDisplay.length > 0;
                }

                return displayItem;
            });

            if (itemsToDisplay.length > 0) {
                //Remove all dividers from beggining of array
                while (itemsToDisplay[0] && itemsToDisplay[0].Divider)
                    itemsToDisplay.shift();
            }

            if (itemsToDisplay.length > 1) {
                contextMenu.append($("<div />", { "class": "button icon-only dropdown-toggle", "title": this._getLocalizableText("ContextMenu") }).append($("<span />", { "class": "fa fa-lg fa-gear" })).append($("<span />", { "class": "icon-right fa fa-caret-down" }))).addClass("button-group");

                var contextMenuItems = $("<ul />", { "class": "dropdown-menu open-up" });

                _.each(itemsToDisplay, function (item, index, list) {
                    var attributes = {};

                    if (item.Divider) {
                        var nextItem;

                        if (index < list.length - 1)
                            nextItem = list[index + 1];

                        if (nextItem && !nextItem.Divider) {
                            attributes["class"] = "divider";
                            contextMenuItems.append($("<li />", attributes));
                        }
                    }
                    else {
                        attributes["class"] = "v-contextMenuItem";
                        attributes["data-action"] = item.Action;

                        if (item.Params) {
                            attributes["data-params"] = JSON.stringify(item.Params);
                        }

                        if (item.TriggerUpdate) {
                            attributes["data-triggerupdate"] = true;
                        }

                        var menuItem = $("<li />", attributes).append($("<a />").append($("<span />", { "class": "item-label", "text": item.Label })));

                        if (item.SubMenuItemsToDisplay && item.SubMenuItemsToDisplay.length > 0) {
                            $("a", menuItem).append($("<span />", { "class": "fa fa-caret-right submenu-icon" }));

                            var subMenu = $("<ul />", { "class": "dropdown-submenu left" });

                            _.each(item.SubMenuItemsToDisplay, function (subMenuItem) {
                                var attributes = { "class": "v-contextMenuItem", "data-action": subMenuItem.Action };

                                if (subMenuItem.Params) {
                                    attributes["data-params"] = JSON.stringify(subMenuItem.Params);
                                }

                                subMenu.append($("<li />", attributes).append($("<a />", { "text": subMenuItem.Label })));
                            });

                            menuItem.append(subMenu);
                        }

                        contextMenuItems.append(menuItem);
                    }

                    previousItem = item;
                });

                contextMenu.append(contextMenuItems);
            }
            else if (itemsToDisplay.length > 0) {
                var item = _.first(itemsToDisplay);
                var attributes = { "data-action": item.Action };
                if (item.Params) {
                    attributes["data-params"] = JSON.stringify(item.Params);
                }
                contextMenu.append(item.Label).addClass("button v-contextMenuItem").attr(attributes);
            }

            return contextMenu.outerHTML();
        },
        _renderToolBar: function () {
            var toolBar = $(".toolBar", this.container);
            var toolBarItems = _.filter(this.options.toolBarItems, function (item) { return item.Visible != false });

            for (var i = 0; i < toolBarItems.length; i++) {
                toolBarItems[i].Position = i;
                var item = toolBarItems[i];
                var attributes = { "class": "button", "data-action": item.Action, "data-position": i };

                if (item.TriggerUpdate) {
                    attributes["data-triggerupdate"] = true;
                }

                var button = $("<div />", attributes).append(item.Label);

                if (!item.DisplayMode || item.DisplayMode == dataTables.itemDisplayMode.selected) {
                    button.hide();
                }

                toolBar.append(button);
            };

            $(".dataTables_filter", this.container).append($("<span />", { "class": "h-filterButton filter-button fa fa-search" }));
            $(".dataTables_filter :input", this.container).attr("placeholder", this._getLocalizableText("Search"));
        },
        _renderQuickFilter: function () {
            var filter = $(".filter-bar", this.container);

            if (filter.length > 0) {
                var filterItems = this.options.quickFilter.Items;
                var filterItemsLength = filterItems.length + 1 + (this.options.quickFilter.IncludeSelectedStates ? 2 : 0);

                var buttonGroup = $("<div />", { "class": "button-group" });

                if ((this.options.quickFilter.DisplayMode == undefined && filterItemsLength <= dataTableConstants.maxFilterLength) || this.options.quickFilter.DisplayMode == dataTables.filterDisplayMode.buttons) {
                    filter.addClass("buttons");

                    if (this.options.quickFilter.IncludeAll !== false) {
                        buttonGroup.append($("<div />", { "class": "button" }).append(this._getLocalizableText("All")));
                    }

                    if (this.options.quickFilter.IncludeSelectedStates) {
                        buttonGroup.append($("<div />", { "class": "button v-selected" }).append(this._getLocalizableText("Selected")));
                        buttonGroup.append($("<div />", { "class": "button v-notSelected" }).append(this._getLocalizableText("NotSelected")));
                    }

                    for (var i = 0; i < filterItems.length; i++) {
                        var button = $("<div />", { "class": "button v-quickFilterButton" }).append(filterItems[i].Label).attr("data-index", i);
                        buttonGroup.append(button);
                    }

                    $(".button:first", buttonGroup).addClass("selected");
                }
                else {
                    filter.addClass("dropdown");

                    buttonGroup.append($("<div />", { "class": "button icon-only dropdown-toggle" }).append($("<span />", { "class": "icon filter" })).append($("<span />", { "class": "label", "text": this._getLocalizableText("All") })).append($("<span />", { "class": "icon-small down-arrow-black" })));

                    var menuItems = $("<ul />", { "class": "dropdown-menu left-align" });

                    if (this.options.quickFilter.IncludeAll !== false) {
                        menuItems.append($("<li />").append($("<a />", { "text": this._getLocalizableText("All") })));
                    }

                    if (this.options.quickFilter.IncludeSelectedStates) {
                        buttonGroup.append($("<li />", { "class": "v-selected" }).append($("<a />", { "text": this._getLocalizableText("Selected") })));
                        buttonGroup.append($("<li />", { "class": "v-notSelected" }).append($("<a />", { "text": this._getLocalizableText("NotSelected") })));
                    }

                    for (var i = 0; i < filterItems.length; i++) {
                        var listItem = $("<li />", { "class": "v-quickFilterButton", "data-index": i }).append($("<a />", { "text": filterItems[i].Label }));
                        menuItems.append(listItem);
                    }

                    $("li:first", menuItems).addClass("selected");

                    buttonGroup.append(menuItems);
                }

                filter.append(buttonGroup);
            }
        },
        _toggleQuickFilter: function (container) {
            if (this.options.quickFilter) {
                var filter = $(".filter-bar", container);

                var filterItems = this.options.quickFilter.Items;

                $(".v-quickFilterButton", filter).each(function () {
                    var filterIndex = $(this).attr("data-index");

                    if (filterItems[filterIndex]) {
                        var visible = a4.evaluateCondition(filterItems[filterIndex].DisplayCondition);
                        $(this).toggleClass("hidden", !visible);
                    }
                });

                $(".button:not(.hidden):first", filter).addClass("first-button");
                $(".button:not(.hidden):last", filter).addClass("last-button");
            }
        },
        _generateEnumControl: function (enumType, selectedValue, nullable) {
            var enumType = EnumDetails[enumType];
            var dropDown = $("<select />", {});

            if (nullable !== false) {
                dropDown.append($("<option />", { "value": null }));
            }

            _.each(enumType, function (e) {
                var option = $("<option />", { "value": e.Value }).append(e.Label);

                if (e.Value == selectedValue || e.Name == selectedValue || e.Label == selectedValue) {
                    option.attr("selected", "selected");
                }

                dropDown.append(option);
            });

            return dropDown;
        },
        _generateBoolFilterCondition: function (labels) {
            var dropDown = $("<select />", {});
            var trueLabel, falseLabel;

            if (labels) {
                trueLabel = labels["1"];
                falseLabel = labels["0"];
            }
            else {
                trueLabel = this._getLocalizableText("Yes");
                falseLabel = this._getLocalizableText("No");
            }

            dropDown.append($("<option />", { "value": null }));
            dropDown.append($("<option />", { "value": 1 }).append(trueLabel));
            dropDown.append($("<option />", { "value": 0 }).append(falseLabel));

            return dropDown;
        },
        _generateDateTimeFilterCondition: function () {
            var div = $("<div />", { "class": "date-filter" });

            div.append($("<input />", { "type": "text", "class": "v-fromDate", "placeholder": this._getLocalizableText("From") }).datetimepicker());
            div.append($("<input />", { "type": "text", "class": "v-toDate", "placeholder": this._getLocalizableText("To") }).datetimepicker());

            return div;
        },
        _renderAdvancedFilterRow: function (header) {
            if (this.options.advancedFilter) {
                var self = this;
                var advancedFilterRow = $("<tr />", { "class": "advanced-filter" });

                _.each(this.columns, function (column) {
                    var filterColumn = $("<th />", { "class": "advanced-filter-column" });

                    if (column.IsDataColumn && column.AdvancedFilter !== true) {
                        filterColumn.attr("data-column", column.name);

                        var columnType = (column.type !== undefined ? column.type : "text").toLowerCase();

                        if (columnType == "enum" && EnumDetails !== undefined) {
                            filterColumn.append(self._generateEnumControl(column.EnumType));
                        }
                        else if (columnType == "boolean") {
                            filterColumn.append(self._generateBoolFilterCondition(column.AdvancedFilterLabels));
                        }
                        else if (columnType == "date" || columnType == "datetime") {
                            filterColumn.append(self._generateDateTimeFilterCondition());
                        }
                        else {
                            var input = $("<input />", { "type": "text" });
                            if (columnType == "numeric") {
                                input.autoNumeric({ aSep: '', aDec: '.', aPad: false, vMin: dataTableConstants.numericMinValue, vMax: dataTableConstants.numericMaxValue });
                            }
                            filterColumn.append(input);
                        }
                    }

                    advancedFilterRow.append(filterColumn);
                });

                var clearButton = $("<span />", { "class": "h-clearAdvancedFilter fa fa-lg fa-times", "title": this._getLocalizableText("ClearFilter") });
                var refreshButton = $("<span />", { "class": "h-applyAdvancedFilter fa fa-lg fa-check", "title": this._getLocalizableText("ApplyFilter") });

                $("th:last", advancedFilterRow).append(refreshButton).append(clearButton);

                header.prepend(advancedFilterRow);
            }
        },
        _bindEvents: function () {
            var self = this;
            var dataTable = this.dataTable;
            var options = this.options;

            dataTable.on("length.dt", function (e, settings, len) {
                self._saveState();
            });

            if (self.allowRowSelection) {
                dataTable.on("change", "tr .selectRow", function () {
                    var table = $(this).closest(".dataTable");
                    var isChecked = $(this).is(":checked");
                    var row = $(this).closest("tr");
                    var rowData = dataTable.row(row).data();

                    table.find(".h-selectAll").prop("checked", false);
                    self._updateSelectedRows(rowData, isChecked);
                    self._setSelectedRowsLabel();
                    self._checkSelectedRowsLimit();

                    row.toggleClass("selected", isChecked);
                    $(".sprite-icons", row).toggleClass('inverted', isChecked);
                    self._toggleToolbarItems(self.container);
                    self._trigger("selectRow", null, { row: row, data: rowData, isChecked: isChecked, selectedRows: self.getSelectedItems() });
                });

                dataTable.on("change", ".h-selectAll", function () {
                    var rows = $(this).closest(".dataTable").find("tbody tr:has(.selectRow)");
                    var rowsData = dataTable.rows(rows).data().toArray();

                    if ($(".dataTables_empty", rows).length == 0) {
                        var isChecked = $(this).is(":checked");
                        rows.toggleClass("selected", isChecked);
                        $(".sprite-icons", rows).toggleClass('inverted', isChecked);

                        _.each(rowsData, function (value) { self._updateSelectedRows(value, isChecked); });
                        self._setSelectedRowsLabel();

                        $(".selectRow", rows).prop("checked", isChecked);
                        self._toggleToolbarItems(self.container);
                        self._trigger("selectAll", null, { rows: rows, data: rowsData, isChecked: isChecked, selectedRows: self.getSelectedItems() });
                    }
                });
            }

            if (options.contextMenuItems.length > 0) {
                dataTable.on("mouseenter", "tbody tr", function () {
                    var contextMenu = $(this).find(".action-menu-column .button-group");

                    if (contextMenu.length > 0) {
                        var menuTop = contextMenu.offset().top;
                        var menuBottom = menuTop + contextMenu.outerHeight();
                        var menuHeight = $(".dropdown-menu", contextMenu).outerHeight(true);

                        var openUp = (menuTop - menuHeight) > 0 && (menuBottom + menuHeight) > $(document).outerHeight();

                        $(".dropdown-menu", contextMenu).toggleClass("open-up", openUp);
                    }
                });

                dataTable.on("click", ".v-contextMenuItem", function (e) {
                    e.stopPropagation();
                    var action = $(this).attr("data-action");

                    if (action) {
                        var rowSelector = $(this).closest("tr");
                        var row = dataTable.row(rowSelector);
                        var rowIndex = row.index();
                        var rowData = row.data();
                        var params = $(this).attr("data-params");

                        if (params)
                            params = JSON.parse(params);

                        self._trigger("contextMenuClick", null, { action: action, params: params, rowSelector: rowSelector, row: rowData });

                        if ($(this).attr("data-triggerupdate")) {
                            self._setRowUpdated(rowSelector, rowData, rowIndex);
                        }
                    }
                });
            }

            if (options.toolBarItems.length > 0) {
                $("div.toolBar div.button", this.container).click(function () {
                    if (!$(this).is(".disabled")) {
                        var params = {
                            action: $(this).attr("data-action")
                        };

                        if (options.rememberRowSelection) {
                            params.rows = self.selectedRows;
                        }
                        else {
                            var rows = dataTable.rows($(".selected"));

                            params.rows = rows.data().toArray();
                            params.rowsSelector = rows.nodes().toArray();
                        }

                        self._trigger("toolBarClick", null, params);
                    }
                });
            }

            if (options.quickFilter) {
                $("div.filter-bar.buttons .button, div.filter-bar.dropdown li", this.container).click(function () {
                    if (!$(this).hasClass("selected")) {
                        var container = $(this).closest(".button-group");

                        $(".selected", container).removeClass("selected");
                        $(this).addClass("selected");

                        var filterIndex = $(this).attr("data-index");
                        self._applyFilter(filterIndex);

                        $(".dropdown-toggle .label", container).text($(this).text());
                    }
                });
            }

            dataTable.on("mouseenter", "td.tooltip-column", function () {
                var tooltipData = options.tooltip.Data;
                var tooltip = $(this).find("div.tooltip-content");

                if (_.isFunction(tooltipData)) {
                    var row = $(this).closest("tr");
                    var rowData = dataTable.row(row).data();
                    tooltipData(tooltip, rowData);
                }

                tooltip.show();
            });

            dataTable.on("mouseleave", "td.tooltip-column", function () {
                $(this).find("div.tooltip-content").hide();
            });

            //Editable columns
            dataTable.on("change textChanged", ".editable", function () {
                var dataProperty = $(this).attr("data-dataproperty");
                if (dataProperty) {
                    var rowSelector = $(this).closest("tr");
                    var value;

                    if ($(this).is("input:checkbox"))
                        value = $(this).is(':checked');
                    else if ($(this).is("div"))
                        value = $(this).html();
                    else if ($(this).hasClass("autocomplete"))
                        value = $(this).attr("data-value");
                    else
                        value = $(this).val();

                    var row = dataTable.row(rowSelector);
                    var rowData = row.data();
                    var rowIndex = row.index();

                    rowData[dataProperty] = value;

                    self._setRowUpdated(rowSelector, rowData, rowIndex);
                }
            });

            //Settings menu
            $(".settings-menu li :checkbox", this.container).change(function () {
                var check = !$(this).is(":checked");
                $(this).prop("checked", check);
                self.toggleColumn($(this).closest("li").attr("data-column-name"), check);
                self.settingsChanged = true;
            });

            $(".settings-menu li", this.container).click(function () {
                var checkbox = $(this).find(":checkbox");
                checkbox.change();
            });

            $(".settings-menu .h-densitySelector .button", this.container).click(function () {
                if (!$(this).hasClass("selected")) {
                    var styleSelector = $(this).closest(".h-densitySelector");

                    var previousStyle = $(".button.selected", styleSelector);

                    previousStyle.removeClass("selected");
                    $("table", this.container).removeClass(previousStyle.attr("data-display-density"));

                    $(this).addClass("selected");
                    $("table", this.container).addClass($(this).attr("data-display-density"));

                    self.settingsChanged = true;
                }
            });

            $(".settings-menu", this.container).mouseenter(function () {
                self.settingsChanged = false;
            });

            $(".settings-menu", this.container).mouseleave(function () {
                if (self.settingsChanged) {
                    self._saveState();
                }
            });

            if (this.options.displaySearchBox) {
                $(".dataTables_filter input", this.container).unbind();

                $(".dataTables_filter input", this.container).keypress(function (e) {
                    if (e.keyCode == 13) {
                        self._filterData();
                    }
                    return e.keyCode != 13;
                });

                $(".h-filterButton", this.container).click(function () {
                    self._filterData();
                });
            }

            $(".h-refreshButton", this.container).click(function () {
                $(this).addClass("fa-spin");
                dataTable.draw(false);
            });

            if (this.options.advancedFilter) {
                $(".advanced-filter-toggle .button", this.container).click(function () {
                    $(this).toggleClass("selected");
                    $(this).find("span").toggleClass("filter filter-white");

                    $("tr.advanced-filter", self.container).toggle();

                    if (!$(this).hasClass("selected")) {
                        self._clearAdvancedFilter();
                    }
                });

                dataTable.on("keypress", ".advanced-filter-column :text", function (e) {
                    if (e.keyCode == 13) {
                        self._applyAdvancedFilter();
                    }
                    return e.keyCode != 13;
                });

                $(".h-applyAdvancedFilter", this.container).click(function () {
                    self._applyAdvancedFilter();
                });

                $(".h-clearAdvancedFilter", this.container).click(function () {
                    self._clearAdvancedFilter();
                });
            }

            if (this.options.multiEdit) {
                $(".multi-edit-toggle .button", this.container).click(function () {
                    self._toggleMultiEdit(!self.editing);
                });

                $(".multi-edit-bar .save", this.container).click(function () {
                    if (self.updatedRows && self.updatedRows.length > 0) {
                        self._trigger("rowsUpdated", null, {
                            rows: _.pluck(self.updatedRows, "data"), multiEditSaveCallback: function () {
                                self._toggleMultiEdit(false);
                            }
                        });
                    }
                    else {
                        self._toggleMultiEdit(false);
                    }
                });

                $(".multi-edit-bar .cancel", this.container).click(function () {
                    self._toggleMultiEdit(false);
                });
            }

            $(window).resize(function () {
                self._adjustColumnHeaders(dataTable);
            });
        },
        _applyAdvancedFilter: function () {
            this.dataTable.draw();
            this.advancedFilterApplied = true;
        },
        _clearAdvancedFilter: function () {
            $("tr.advanced-filter :input", this.container).val("");

            if (this.advancedFilterApplied) {
                this.dataTable.draw();
                this.advancedFilterApplied = false;
            }
        },
        _initializeSelectedRows: function () {
            var keyColumn = this.keyColumn;
            if (this.options.rememberRowSelection) {
                var selectedRows = this.options.selectedRows;
                if (selectedRows) {
                    if (_.isArray(selectedRows)) {
                        this.selectedRows = selectedRows;
                    }
                    else {
                        this.selectedRows = _.map(selectedRows.split(","), function (item) {
                            var row = {};
                            row[keyColumn] = item;
                            return row;
                        });
                    }
                }
                else {
                    this.selectedRows = [];
                }

                this._setSelectedRowsLabel();
            }
        },
        _updateSelectedRows: function (rowData, isChecked) {
            var keyColumn = this.keyColumn;
            var key = rowData[keyColumn];

            if (this.options.rememberRowSelection && keyColumn) {
                if (isChecked) {
                    if (!_.some(this.selectedRows, function (item) { return item[keyColumn] == key; })) {
                        this.selectedRows.push(rowData);
                    }
                }
                else {
                    this.selectedRows = _.reject(this.selectedRows, function (item) { return item[keyColumn] == key; });
                }
            }
        },
        _checkSelectedRowsLimit: function () {
            if (this.selectedRows && this.options.limitRowSelection) {
                if (this.selectedRows.length >= this.options.limitRowSelection) {
                    $("tr .selectRow:not(:checked)", this.container).prop("disabled", true);
                }
                else {
                    $("tr .selectRow", this.container).prop("disabled", false);
                }
            }
        },
        _setSelectedRowsLabel: function () {
            var label = "";

            if (this.selectedRows && this.selectedRows.length > 0) {
                if (this.options.limitRowSelection) {
                    label = this.selectedRows.length == 1 ? this._getLocalizableText("SelectedItemLimit") : this._getLocalizableText("SelectedItemsLimit");
                }
                else {
                    label = this.selectedRows.length == 1 ? this._getLocalizableText("SelectedItem") : this._getLocalizableText("SelectedItems");
                }

                label = "(" + label.replace(/{count}/g, this.selectedRows.length).replace(/{limit}/g, this.options.limitRowSelection) + ")";
            }

            $(".v-selectedRows", this.container).text(label);
        },
        _filterData: function () {
            var filterValue = $(".dataTables_filter input", this.container).val();
            this.dataTable.search(filterValue).draw();
        },
        _renderAdvancedFilterToggleButton: function () {
            if (this.options.advancedFilter) {
                var advancedFilterToggle = $("<div />", { "class": "button icon-only", "title": this._getLocalizableText("ToggleAdvancedFilter") }).append($("<span />", { "class": "fa fa-filter" }));
                $(".advanced-filter-toggle", this.container).append(advancedFilterToggle);
            }
        },
        _renderMultiEditControls: function () {
            if (this.options.multiEdit) {
                var multiEditToggle = $("<div />", { "class": "button icon-only", "title": this._getLocalizableText("ToggleMultiEdit") }).append($("<span />", { "class": "fa fa-pencil" }));
                $(".multi-edit-toggle", this.container).append(multiEditToggle);

                var multiEditBar = $('.multi-edit-bar', this.container);
                multiEditBar.append($("<div />", { "class": "link-button cancel small" }).append(this._getLocalizableText("Cancel")));
                multiEditBar.append($("<div />", { "class": "button save primary small" }).append(this._getLocalizableText("SaveChanges")));
            }
        },
        _renderSettingsMenu: function () {
            var settingsMenu = $(".settings-menu", this.container);

            if (settingsMenu.length > 0) {
                var buttonGroup = $("<div />", { "class": "button-group" });
                var self = this;

                buttonGroup.append($("<div />", { "class": "button icon-only dropdown-toggle" }).append($("<span />", { "class": "fa fa-lg fa-columns" })).append($("<span />", { "class": "icon-right fa fa-caret-down" }))).addClass("button-group");

                var menu = $("<ul />", { "class": "dropdown-menu keep-open" });

                _.each(this.options.columns, function (item) {
                    if (item.Data && !item.HideInSelector && !item.Tooltip) {
                        var checkbox = $("<input type='checkbox'>").prop("checked", self._isVisibleColumn(item));
                        menu.append($("<li />", { "data-column-name": item.Name }).append($("<a />", { "html": item.Title }).prepend(checkbox)));
                    }
                });

                buttonGroup.append(menu);

                settingsMenu.append(buttonGroup);
            }
        },
        _getTableParams: function (aoData) {
            var data = {};

            if (aoData) {
                data.Echo = aoData.draw;
                data.PageStart = aoData.start;
                data.PageLength = aoData.length;

                if (aoData.search) {
                    data.Search = aoData.search.value;
                    data.IsSearch = data.Search != "";
                }

                if (aoData.order && aoData.order.length > 0) {
                    var column = aoData.columns[aoData.order[0].column];
                    data.SortColumn = column.name;
                    data.SortAscending = aoData.order[0].dir != "desc";
                    data.IsSort = true;
                }

                if (this.options.quickFilter && this.options.quickFilter.IncludeSelectedStates && ($(".v-selected.selected").length > 0 || $(".v-notSelected.selected").length > 0)) {
                    data.Selected = "'" + _.pluck(this.selectedRows, this.keyColumn).join("','") + "'";
                    data.ExcludeSelected = ($(".v-notSelected.selected").length > 0);
                }

                if (this.options.advancedFilter) {
                    data.AdvancedFilter = this._getTableAdvancedFilterParam();
                }
            }

            return data;
        },
        _getTableAdvancedFilterParam: function () {
            var self = this;
            var advancedFilterRow = $("tr.advanced-filter", this.container);
            var advancedFilter = [];

            if (advancedFilterRow.is(":visible")) {
                $(".advanced-filter-column[data-column]", advancedFilterRow).each(function () {
                    var column = self._getColumnByName($(this).attr("data-column"));

                    if (column.type == "date" || column.type == "datetime") {
                        var fromDate = $(this).find(".v-fromDate").val();
                        var toDate = $(this).find(".v-toDate").val();

                        if (fromDate || toDate) {
                            advancedFilter.push({ "Name": column.name, "Value": fromDate, "OtherValue": toDate, "Type": column.type });
                        }
                    }
                    else {
                        var value = $(this).find(":input").val();

                        if (value) {
                            advancedFilter.push({ "Name": column.name, "Value": value, "Type": column.type });
                        }
                    }
                });
            }

            return advancedFilter;
        },
        _toggleMultiEdit: function (editing) {
            var self = this;

            this.editing = editing;
            this._clearUpdatedRows();

            self.dataTable.column("SelectRow:name, ActionsMenu:name").visible(!editing);

            //Do editing operations for the next draw            
            this.dataTable.one("draw", function () {
                $('.multi-edit-bar', self.container).toggle(editing);
                $('.multi-edit-toggle .button', self.container).toggleClass("selected", editing);
                $('.multi-edit-toggle .button span', self.container).toggleClass("edit-white", editing);
                $('.multi-edit-toggle .button span', self.container).toggleClass("edit", !editing);
            });

            this.dataTable.draw(false);
        },
        _setEditableItems: function () {
            $(".editable[data-type=numeric]", this.container).not(".hasAutoNumeric").addClass("hasAutoNumeric").autoNumeric({ aSep: '', aDec: '.', aPad: false, vMin: dataTableConstants.numericMinValue, vMax: dataTableConstants.numericMaxValue });

            //$(".editable[data-type=datetime], .editable[data-type=date]", this.container).not(".hasDatepicker").datetimepicker();

            //$(".editable.v-qeRichEditText", this.container).not(".ui-editable-label").makeEditable({ PreventInitialTextChanged: true });
        },
        _setRowUpdated: function (row, rowData, rowIndex) {
            var updatedRow = _.find(this.updatedRows, function (r) { return r.index == rowIndex; });

            if (updatedRow) {
                updatedRow.data = rowData;
            }
            else {
                this.updatedRows.push({ index: rowIndex, data: rowData });
            }

            this._trigger("rowUpdated", null, { row: row, data: rowData });
        },
        _clearUpdatedRows: function () {
            this.updatedRows = [];
        },
        _applyFilter: function (filterIndex) {
            if (this.options.quickFilter) {
                var filter = this.options.quickFilter.Items[filterIndex];

                //Reset the params to what is specified by the user for the grid
                this.ajaxParams = _.clone(this.options.ajaxParams);

                //Merge grid params with filter params
                if (filter && filter.AjaxParams) {
                    this.ajaxParams = _.extend(this.ajaxParams, filter.AjaxParams);
                }
                else {
                    this.ajaxParams.filterType = 0;
                }

                this.refresh(true);

                this._trigger("applyFilter", null, { filter: filter });
            }
        },
        clear: function () {
            this.dataTable.clear().draw();
        },
        refresh: function (resetPagination, resetSelectedRows) {
            if (resetSelectedRows) {
                this.selectedRows = [];
            }

            this.dataTable.draw(resetPagination || false);
        },
        setPage: function (pageNumber) {
            var self = this;
            if (self.setPageTrialCount == undefined) {
                self.setPageTrialCount = 10;
            }
            else {
                self.setPageTrialCount -= 1;
            }

            if (self.setPageTrialCount > 0) {
                var pageInfo = self.dataTable.page.info();
                if (pageInfo.pages > 0) {
                    self.dataTable.page(pageNumber).draw('page');
                    //console.log("setPage> datable set page done");
                }
                else { // Page info not yet loaded.  Wait to make sure page api will work
                    setTimeout(function () {
                        //console.log("setPage> datable not load yet, try again later...Trial count = " + self.setPageTrialCount);
                        self.setPage(pageNumber);
                    }, 500);
                }
            }
        },
        getDisplayedRecords: function () {
            return this.dataTable.page.info().recordsDisplay;
        },
        getTotalRecords: function () {
            return this.dataTable.page.info().recordsTotal;
        },
        getSelectedColumns: function () {
            return $(".settings-menu", this.container).find(":checkbox:checked").map(function () { return $(this).closest("li").attr("data-column-name"); }).get();
        },
        getVisibleColumns: function () {
            return _.pluck(_.filter(this.dataTable.settings()[0].aoColumns, function (c) { return c.IsDataColumn && c.bVisible }), "sName");
        },
        toggleColumn: function (columnName, showOrHide) {
            var column = this.dataTable.column(columnName + ":name");

            if (showOrHide == undefined) {
                showOrHide = !column.visible();
            }

            column.visible(showOrHide);

            $(".settings-menu li[data-column-name='" + columnName + "'] :checkbox", this.container).prop("checked", showOrHide);

            //Update empty row colspan
            $("td.dataTables_empty", this.container).attr("colspan", $("thead tr.header-row th", this.container).length);
        },
        changeColumnTitle: function (columnName, columnTitle) {
            var header = this.dataTable.column(columnName + ":name").header();
            $(".column-title", header).text(columnTitle);
        },
        toggleToolBarItem: function (itemAction, showOrHide) {
            var toolBarItem = _.find(this.options.toolBarItems, function (item) { return item.Action == itemAction; });
            if (toolBarItem) {
                if (showOrHide == undefined) {
                    showOrHide = !toolBarItem.Disabled;
                }

                toolBarItem.Disabled = !showOrHide;
            }
        },
        getSelectedItems: function () {
            var selectedRows = [];

            if (this.dataTable) {
                if (this.options.rememberRowSelection) {
                    selectedRows = this.selectedRows;
                }
                else {
                    var rows = this.dataTable.rows($(".selected"));
                    selectedRows = rows.data().toArray();
                }
            }

            return selectedRows;
        },
        getCurrentPage: function () {
            return this.dataTable.page.info().page;
        },
        setCurrentPage: function (page, callback) {
            if (callback) {
                this.dataTable.one("draw", callback);
            }

            this.dataTable.page(page).draw(false);
        },
        clearTable: function () {
            this.dataTable.clear().draw();
        },
        deleteRow: function (row) {
            this.dataTable.row(row).remove().draw(false);
        },
        getNodes: function () {
            return this.dataTable.rows().nodes();
        },
        getNodesData: function () {
            return this.dataTable.rows().data();
        },
        getNodeData: function (i) {
            return this.dataTable.rows().data()[i];
        },
        getNode: function (i) {
            return this.dataTable.rows().nodes()[i];
        },
        setSelectedRow: function (row, rowData) {
            var self = this;
            var isChecked = $('.selectRow', row).is(":checked");

            self._updateSelectedRows(rowData, isChecked);
            self._setSelectedRowsLabel();
            self._checkSelectedRowsLimit();

            $(row).toggleClass("selected", isChecked);
            $(".sprite-icons", row).toggleClass('inverted', isChecked);
            self._toggleToolbarItems(self.container);
        },
        resetOtherSelectedRows: function (row,rowData) {
            var self = this;
            var selectedRows = this.selectedRows;
            
            if (this.dataTable) {
                var selectedRowsToDelete = _.difference(this.selectedRows, [rowData]);
                $.each(selectedRowsToDelete, function(i, e) {
                    selectedRows = _.without(selectedRows, e);
                });
            }
            this.selectedRows = selectedRows;
            self._setSelectedRowsLabel();
            self._checkSelectedRowsLimit();
            self._toggleToolbarItems(self.container);
        },
        resetSearch: function () {
            if (this.dataTable) {
                this.dataTable.search('');
            }
        },
        getSelectedItemsCount: function () {
            if (this.dataTable) {
                if (this.options.rememberRowSelection) {
                    return this.selectedRows.length;
                }
                else {
                    var rows = this.dataTable.rows($(".selected"));
                    return  rows.data().length;
                }
            }
            return 0;
        },
        getTableParams: function () {
            var tableParams = null;

            if (this.dataTable) {
                tableParams = this._getTableParams(this.dataTable.ajax.params());
            }

            return tableParams;
        },
        getAjaxParams: function () {
            var params = this.getTableParams();

            if (this.ajaxParams) {
                params = _.extend(params, this.ajaxParams);
            }

            return params;
        },
        getEditingState: function () {
            var self = this;
            return self.editing;
        },
        hideQuickFilter: function () {
            if (this.options.quickFilter) {
                $('.filter-bar.buttons').hide();
            }
        },
        showQuickFilter: function (filters) {
            if (this.options.quickFilter) {
                $('.filter-bar.buttons').show();
            }
        },
        _adjustColumnHeaders: function (api) {
            setTimeout(function () {
                api.columns.adjust();
            }, 0);
        }
    });
}(jQuery));